<?php
// get_home_data.php (FINAL - HIGH-PERFORMANCE & is_active FILTERING DISABLED)

header('Content-Type: application/json; charset=utf-8');
require 'db_connect.php'; 

$response = array();
$response['status'] = 'error';
$response['message'] = 'An unknown error occurred.';

if (!isset($conn) || $conn->connect_error) {
    http_response_code(500);
    $response['message'] = 'Server Error: Database connection failed.';
    echo json_encode($response);
    exit;
}

$conn->set_charset("utf8mb4");

try {
    $all_product_ids = [];
    $product_map = [];

    // --- 1. Fetch Trending Products (NO is_active filter) ---
    $products = [];
    $product_sql = "SELECT id, name, regular_price, offer_price, image_url, is_featured, is_active FROM products ORDER BY id DESC LIMIT 150";
    $product_result = $conn->query($product_sql);
    if ($product_result) {
        while ($row = $product_result->fetch_assoc()) {
            $product_id = (int)$row['id'];
            $products[] = $row;
            if (!isset($product_map[$product_id])) {
                $all_product_ids[] = $product_id;
                $product_map[$product_id] = $row;
            }
        }
    }
    
    // --- 2. Fetch Featured Products (NO is_active filter) ---
    $featured_products = [];
    $featured_sql = "SELECT id, name, regular_price, offer_price, image_url, is_featured, is_active FROM products WHERE is_featured = 1 ORDER BY id DESC LIMIT 10";
    $featured_result = $conn->query($featured_sql);
    if ($featured_result) {
        while ($row = $featured_result->fetch_assoc()) {
             $product_id = (int)$row['id'];
            $featured_products[] = $row;
            if (!isset($product_map[$product_id])) {
                $all_product_ids[] = $product_id;
                $product_map[$product_id] = $row;
            }
        }
    }

    // --- 3. OPTIMIZED: Fetch all ratings in a single query ---
    $ratings_map = [];
    $unique_product_ids = array_unique($all_product_ids);
    if (!empty($unique_product_ids)) {
        $ids_string = implode(',', $unique_product_ids);
        $ratings_sql = "
            SELECT product_id, AVG(rating) as rating, COUNT(id) as total_reviews
            FROM reviews
            WHERE product_id IN ($ids_string) AND status = 'Published'
            GROUP BY product_id";
        $ratings_result = $conn->query($ratings_sql);
        if ($ratings_result) {
            while ($row = $ratings_result->fetch_assoc()) {
                $ratings_map[(int)$row['product_id']] = $row;
            }
        }
    }
    
    // --- 4. Merge ratings back into product lists ---
    $final_products = [];
    foreach ($products as $product) {
        $id = (int)$product['id'];
        $product['rating'] = isset($ratings_map[$id]) ? (float)$ratings_map[$id]['rating'] : 0.0;
        $product['total_reviews'] = isset($ratings_map[$id]) ? (int)$ratings_map[$id]['total_reviews'] : 0;
        $final_products[] = $product;
    }

    $final_featured_products = [];
    foreach ($featured_products as $product) {
        $id = (int)$product['id'];
        $product['rating'] = isset($ratings_map[$id]) ? (float)$ratings_map[$id]['rating'] : 0.0;
        $product['total_reviews'] = isset($ratings_map[$id]) ? (int)$ratings_map[$id]['total_reviews'] : 0;
        $final_featured_products[] = $product;
    }
    
    $final_best_deals = []; // Best Deals are disabled for testing.

    // --- 5. Fetch Other Data ---
    $sliders = [];
    $slider_result = $conn->query("SELECT id, image_url FROM sliders ORDER BY id ASC");
    if ($slider_result) { while ($row = $slider_result->fetch_assoc()) { $sliders[] = $row; } }
    
    $categories = [];
    $category_result = $conn->query("SELECT id, name, image_url FROM categories ORDER BY name ASC");
    if ($category_result) { while ($row = $category_result->fetch_assoc()) { $categories[] = $row; } }
    
    $promo_banner = $conn->query("SELECT image_url FROM promo_banner LIMIT 1")->fetch_assoc();
    $notice_row = $conn->query("SELECT notice FROM notices ORDER BY id DESC LIMIT 1")->fetch_assoc();
    $popup_banner = $conn->query("SELECT image_url FROM popup_banners LIMIT 1")->fetch_assoc();
    
    // --- 6. Assemble the final data object ---
    $data = [
        'sliders' => $sliders,
        'categories' => $categories,
        'products' => $final_products,
        'bestDeals' => $final_best_deals,
        'featuredProducts' => $final_featured_products,
        'promoBanner' => $promo_banner,
        'notice' => $notice_row ? $notice_row['notice'] : null,
        'popupBanner' => $popup_banner
    ];

    $response['status'] = 'success';
    $response['message'] = 'Home data fetched successfully.';
    $response['data'] = $data;

} catch (Exception $e) {
    http_response_code(500);
    $response['message'] = 'An exception occurred on the server: ' . $e->getMessage();
}

$conn->close();
echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_NUMERIC_CHECK);
exit;
?>