<?php
// FINAL SCRIPT: get_offer_details.php

// --- 1. Prevent Network Errors & Set Up ---
// Start output buffering to catch any stray whitespace or errors
ob_start();

// Set up error handling for production (log errors, don't display them)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Include your database connection script
require 'db_connect.php'; 

// Set the header to indicate the response will be in JSON format
header('Content-Type: application/json');

// Clean any output that may have occurred before this point (e.g., from db_connect.php)
ob_clean();

$response = [];

// --- 2. Validation ---
// Validate the database connection
if (!isset($conn) || $conn->connect_error) {
    http_response_code(500);
    $response['status'] = 'error';
    $response['message'] = 'Server Error: Database connection is not available.';
    echo json_encode($response);
    exit;
}

// Validate that an 'id' parameter was sent in the request URL
if (!isset($_GET['id']) || empty($_GET['id'])) {
    http_response_code(400); // Bad Request
    $response['status'] = 'error';
    $response['message'] = 'Offer ID is missing.';
    echo json_encode($response);
    exit;
}

// Sanitize the input to ensure it's an integer
$offer_id = intval($_GET['id']);

// --- 3. Prepare and Execute the SQL Query ---
// The query selects all columns for the specified offer ID
$sql = "SELECT id, title, description, start_date, end_date, image_url FROM offers WHERE id = ?";

$stmt = $conn->prepare($sql);

if ($stmt) {
    // Bind the integer ID to the '?' placeholder in the query
    $stmt->bind_param("i", $offer_id);
    
    // Execute the prepared statement
    if ($stmt->execute()) {
        $result = $stmt->get_result();
        
        // Check if a result was found
        if ($result->num_rows > 0) {
            $offer = $result->fetch_assoc();
            
            // --- 4. Success: Send the clean offer data ---
            echo json_encode($offer);
        } else {
            // No offer was found with the provided ID
            http_response_code(404); // Not Found
            $response['status'] = 'error';
            $response['message'] = 'Offer not found.';
            echo json_encode($response);
        }
    } else {
        // The query failed to execute
        http_response_code(500);
        $response['status'] = 'error';
        $response['message'] = 'Query execution failed.';
        echo json_encode($response);
    }
    $stmt->close();
} else {
    // The SQL statement could not be prepared (e.g., syntax error)
    http_response_code(500);
    $response['status'] = 'error';
    $response['message'] = 'Statement preparation failed.';
    echo json_encode($response);
}

// --- 5. Final Clean Output ---
// Get anything that was echo'd into the buffer
$finalOutput = ob_get_contents();
// Clean the buffer
ob_clean();
// Echo the final, clean output
echo $finalOutput;

// Close the connection and stop the script
$conn->close();
exit;
?>